<?php

namespace App\Controller;

use App\Entity\Model\Empresa;
use App\Entity\Model\PlacasSocios;
use App\Entity\Model\TransporteSocio;
use App\Form\ListGenericType;
use App\Form\SearchTransporteSocioType;
use App\Form\SocioTransporteType;
use App\Form\TransportistaType;
use App\Service\EmFactory;
use App\Util\Funciones;
use App\Util\RestApiFunciones;
use Doctrine\Persistence\ManagerRegistry;
use Knp\Component\Pager\PaginatorInterface;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;
use Symfony\Component\Routing\Annotation\Route;
use App\Entity\Model\Transportista;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Validator\Validator\ValidatorInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

/**
 * @Route("/partner")
 */
class SocioTransporteController extends AbstractController
{
    protected $translator;

    /**
     * @Route("", name="partner_transporte_index")
     *
     */
    public function indexAction(EmFactory $emFactory, Request $request, PaginatorInterface $paginator, TranslatorInterface $translator): \Symfony\Component\HttpFoundation\Response
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $repo = $em->getRepository(TransporteSocio::class);
        $repo->setPaginator($paginator);
        // @todo Unhardcode this.
        $limit = 50;

        $form = $this->createForm(SearchTransporteSocioType::class, null, [
            'action' => $this->generateUrl('partner_transporte_index'),
            'method' => 'GET',
        ]);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $pagination = $repo->paginatedSearch($form->getData(), $limit, $request->query->getInt('page', 1), $empresa->getId());
        } else {
            $pagination = $repo->paginatedSearch([], $limit, $request->query->getInt('page', 1), $empresa->getId());
        }

        $transportistas = [];
        foreach ($pagination->getItems() as $item) {
            $transportistas[] = $item;
        }

        $listForm = $this->createForm(ListGenericType::class, $transportistas, [
            'action' => $this->generateUrl('partner_transporte_index'),
        ]);
        $listForm->handleRequest($request);
        if ($listForm->isSubmitted() && $listForm->isValid()) {
            $data = $listForm->getData();
            if ($request->request->has('delete')) {
                if (empty($data['invoices'])) {
                    $this->addTranslatedMessage('Seleccione un registro', 'warning');
                } else {
                    foreach ($data['invoices'] as $transportista) {
                        $em->remove($transportista);
                    }

                    $em->flush();
                    $this->addTranslatedMessage('Registro Eliminado');

                    // Rebuild the query, since some objects are now missing.
                    return $this->redirectToRoute('partner_transporte_index');
                }
            }
        }

        return $this->render('SocioTransporte\index.html.twig',
            [
                'transportistas' => $pagination,
                //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
                'currency' => $empresa == null ? 'USD' : $empresa->getCurrency(),
                'search_form' => $form->createView(),
                'list_form' => $listForm->createView(),
            ]);
    }

    /**
     * @Route("/autocomplete", name="socio_transporte_autocomplete")
     */
    public function autocompleteAction(EmFactory $emFactory, Request $request): \Symfony\Component\HttpFoundation\Response
    {

        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $entities = $em
            ->getRepository(Transportista::class)
            ->findLike($request->get('term'), $empresa->getId());

        return new JsonResponse($entities);
    }

    /**
     * @Route("/add", name="partner_transporte_add")
     *
     */
    public function addAction(EmFactory $emFactory, Request $request, TranslatorInterface $translator, UserPasswordHasherInterface $passwordEncoder): \Symfony\Component\HttpFoundation\Response
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $transportista = new TransporteSocio();
        $placa = new PlacasSocios();
        $transportista->addPlaca($placa);
        $transportista->setEmpresa($empresa);

        $form = $this->createForm(SocioTransporteType::class, $transportista, [
            'action' => $this->generateUrl('partner_transporte_add'),
        ]);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {

            $numDocumento = $transportista->getRucsocio();
            //$respuesta = $transportista->validarRuc($numDocumento, $transportista->getTipoDocumento());
            $respuesta = null;
            if ($respuesta === null) {

                $entities = $em->getRepository(TransporteSocio::class)->findIdentificacion($numDocumento, $empresa);

                if ((is_countable($entities) ? count($entities) : 0) > 0) {
                    $this->addTranslatedMessage('Número de RUC ya se encuentra registrado', 'danger');
                } else {
                    $entities = $em->getRepository(TransporteSocio::class)->findPtoEmision($transportista->getPtoemisionsocio(), $empresa);

                    if ((is_countable($entities) ? count($entities) : 0) > 0) {
                        $this->addTranslatedMessage('Punto de Emisión ya se encuentra registrado', 'danger');
                    } else {

                        $base_api = $this->getParameter('base_api');
                        $msg = Funciones::createNewUser($passwordEncoder, $em, $transportista, $base_api);
                        if ($msg !== null)
                            $this->addFlash('danger', $msg);
                        else {
                            $em->persist($transportista);

                            $emisorTransportista = $empresaRepo->findOneByUserTransportista($transportista->getRucsocio());

                            if (!$emisorTransportista) {
                                $emisorTransportista = Funciones::setNuevaEmpresaTransportista($transportista, $em);
                            } else
                                $emisorTransportista = Funciones::setUpdateEmpresaTransportista($emisorTransportista, $transportista);

                            $em->persist($emisorTransportista);

                            $em->flush();
                            $app_url = $this->getParameter('api_url');
                            $msg = Funciones::setEmisorApi($empresa, null, null, $app_url);
                            if ($msg !== null)
                                $this->addFlash('danger', $msg);

                            $this->addTranslatedMessage('Registro insertado');
                            return $this->redirectToRoute('partner_transporte_index');
                        }

                    }
                }
            } else {
                $this->addTranslatedMessage('Documento del Socio NO Válido ', 'danger');
                //return $this->redirectToRoute('registro_nuevo');
            }
        }

        return $this->render('SocioTransporte/edit.html.twig',
            [
                'form' => $form->createView(),
                'entity' => $transportista
            ]);
    }

    /**
     * @Route("/edit/{slug}tra{id}", name="partner_transporte_edit")
     *
     */
    public function editAction($id, EmFactory $emFactory, Request $request, $slug, TranslatorInterface $translator,  UserPasswordHasherInterface $passwordEncoder): \Symfony\Component\HttpFoundation\Response
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $transportista = $em->getRepository(TransporteSocio::class)->findBySlug($slug, $id);
        if (!$transportista) {
            throw $this->createNotFoundException('Unable to find Transporte Socio entity.');
        }

        $form = $this->createForm(SocioTransporteType::class, $transportista, [
            'action' => $this->generateUrl('partner_transporte_edit', ['slug' => $slug, 'id' => $id])
        ]);
        $form->handleRequest($request);

        if ($form->isSubmitted()) {
            if ($request->request->has('Form-delete')) {
                $em->remove($transportista);
                $em->flush();
                $this->addTranslatedMessage('Registro Eliminado');
                return $this->redirectToRoute('partner_transporte_index');
            }

            if ($form->isValid()) {
                $numDocumento = $transportista->getRucsocio();

                $respuesta = null;

                //if ($transportista->getIdentification() != '9999999999999')
                //    $respuesta = $transportista->validarRuc($numDocumento, $transportista->getTipoDocumento());
                //else

                if ($respuesta === null) {
                    $transportista->setEmpresa($empresa);
                    $em->persist($transportista);

                    $emisorTransportista = $empresaRepo->findOneByUserTransportista($transportista->getRucsocio());

                    if(!$emisorTransportista) {
                        $emisorTransportista = Funciones::setNuevaEmpresaTransportista($transportista, $em);
                    }
                    else
                        $emisorTransportista = Funciones::setUpdateEmpresaTransportista($emisorTransportista, $transportista);

                    $em->persist($emisorTransportista);
                    $em->flush();

                    $base_api = $this->getParameter('base_api');
                    $msg = Funciones::createNewUser($passwordEncoder, $em, $transportista, $base_api);
                    if($msg !== null)
                        $this->addFlash('danger', $msg);
                    else{
                        $app_url = $this->getParameter('api_url');
                        $msg = Funciones::setEmisorApi($empresa, null, null, $app_url);
                        if($msg !== null)
                            $this->addFlash('danger', $msg);
                    }

                    $this->addTranslatedMessage('Registro Actualizado');

                    return $this->redirectToRoute('partner_transporte_edit', ['slug' => $slug, 'id' => $id]);

                } else {
                    $this->addTranslatedMessage('Documento del Socio NO Válido ', 'danger');
                    //return $this->redirectToRoute('registro_nuevo');
                }
            }
        }

        return $this->render('SocioTransporte/edit.html.twig',
            ['form' => $form->createView(), 'entity' => $transportista]);
    }

    protected function addTranslatedMessage($message, $status = 'success')
    {
        $this->addFlash($status, $this->translator->trans($message, [], 'transportista'));

    }

    /**
     * @Route("/searchpartner", name="partner_transporte_searchpartner")
     */
    public function searchSocioAction(Request $request, EmFactory $emFactory): \Symfony\Component\HttpFoundation\Response
    {
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $entities = $em
            ->getRepository(Transportista::class)
            ->findIdentificacion($request->get('term'), $empresa);

        if (count($entities) > 0) {

            if (count($entities) > 1) {
                return new JsonResponse('ERROR, Se encontraron mas de un registro con el número de identificación ingresado',
                    \Symfony\Component\HttpFoundation\Response::HTTP_BAD_REQUEST);
            } else {
                return new JsonResponse(['message' => 'Socio', 'item' => $entities[0]], \Symfony\Component\HttpFoundation\Response::HTTP_OK);
            }

        } else {
            return new JsonResponse(
                'ERROR, Socio no existe'
                , \Symfony\Component\HttpFoundation\Response::HTTP_BAD_REQUEST);
        }

    }


    /**
     * @Route("/search-reference", name="partner_search_reference")
     */
    public function searchReferenceAction(EmFactory $emFactory, Request $request): \Symfony\Component\HttpFoundation\Response
    {
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');

        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $limit = 10;

        $aux = $request->request->all();

        $page = (int) $aux["page"];

        $start = $page > 1 ? ($page - 1) * $limit : 0;

        $condition = Funciones::cleanParametro($aux["query"]);

        $tipo = Funciones::cleanParametro($aux["tipo"]);

        //$entities = $em->getRepository('App\Entity\Model\Inventarios\Product')
        //    ->findxReference($request->get('term'), $empresa->getId());

        $repo = $em->getRepository(Transportista::class);

        $sql = "'p.id','p.reference','p.description'";
        $paginator = $repo->paginatedAjax($sql, $condition, $limit, $start, $empresa->getId());

        $total_data = is_countable($paginator) ? count($paginator) : 0;
        $pagination = [];
        foreach ($paginator as $post) {
            //$post = new Provider();
            $pagination[] = [
                'id' => $post->getIdentification(),
                'nombre' => $post->getName(),
                'email' => $post->getEmail(),
                'placa' => $post->getPlaca(),
                /*'address' => $post->getAddress()*/
            ];
        }

        if ($total_data > 0)
            $pagination_html = Funciones::getPaginationResults($total_data, $limit, $page, $condition, $tipo);
        else
            $pagination_html = '';

        $output = ['data' => $pagination, 'pagination' => $pagination_html, 'total_data' => $total_data];

        return new JsonResponse($output);

    }

}
