<?php

namespace App\Controller;

use App\Form\TransportistaType;
use App\Service\EmFactory;
use App\Util\Funciones;
use App\Util\RestApiFunciones;
use Knp\Component\Pager\PaginatorInterface;
use Symfony\Component\Routing\Annotation\Route;
use App\Entity\Model\Transportista;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Validator\Validator\ValidatorInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

/**
 * @Route("/transportista")
 */
class TransportistaController extends AbstractController
{
    protected $translator;

    /**
     * @Route("", name="transportista_index")
     *
     */
    public function indexAction(EmFactory $emFactory, Request $request, PaginatorInterface $paginator, TranslatorInterface $translator): \Symfony\Component\HttpFoundation\Response
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $repo = $em->getRepository(Transportista::class);
        $repo->setPaginator($paginator);
        // @todo Unhardcode this.
        $limit = 50;

        $form = $this->createForm(\App\Form\SearchTransportistaType::class, null, [
            'action' => $this->generateUrl('transportista_index'),
            'method' => 'GET',
        ]);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $pagination = $repo->paginatedSearch($form->getData(), $limit, $request->query->getInt('page', 1), $empresa->getId());
        } else {
            $pagination = $repo->paginatedSearch([], $limit, $request->query->getInt('page', 1), $empresa->getId());
        }

        $transportistas = [];
        foreach ($pagination->getItems() as $item) {
            $transportistas[] = $item;
        }

        $listForm = $this->createForm(\App\Form\ListGenericType::class, $transportistas, [
            'action' => $this->generateUrl('transportista_index'),
        ]);
        $listForm->handleRequest($request);
        if ($listForm->isSubmitted() && $listForm->isValid()) {
            $data = $listForm->getData();
            if ($request->request->has('delete')) {
                if (empty($data['transportistas'])) {
                    $this->addTranslatedMessage('flash.nothing_selected', 'warning');
                } else {
                    foreach ($data['transportistas'] as $transportista) {
                        $em->remove($transportista);
                    }

                    $em->flush();
                    $this->addTranslatedMessage('Registro Eliminado');

                    // Rebuild the query, since some objects are now missing.
                    return $this->redirectToRoute('transportista_index');
                }
            }
        }

        return $this->render('Transportista\index.html.twig',
            [
                'transportistas' => $pagination,
                //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
                'currency' => $empresa == null ? 'USD' : $empresa->getCurrency(),
                'search_form' => $form->createView(),
                'list_form' => $listForm->createView(),
            ]);
    }

    /**
     * @Route("/autocomplete", name="transportista_autocomplete")
     */
    public function autocompleteAction(EmFactory $emFactory, Request $request): \Symfony\Component\HttpFoundation\Response
    {

        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $entities = $em
            ->getRepository(Transportista::class)
            ->findLike($request->get('term'), $empresa->getId());

        return new JsonResponse($entities);
    }

    /**
     * @Route("/add", name="transportista_add")
     *
     */
    public function addAction(EmFactory $emFactory, Request $request, TranslatorInterface $translator): \Symfony\Component\HttpFoundation\Response
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $transportista = new Transportista();
        $transportista->setEmpresa($empresa);

        $form = $this->createForm(\App\Form\TransportistaType::class, $transportista, [
            'action' => $this->generateUrl('transportista_add'),
        ]);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {

            $numDocumento = $transportista->getIdentification();
            //$respuesta = $transportista->validarRuc($numDocumento, $transportista->getTipoDocumento());
            $respuesta = null;
            if ($respuesta === null) {

                $entities = $em->getRepository(Transportista::class)->findIdentificacion($numDocumento, $empresa);

                if ((is_countable($entities) ? count($entities) : 0) > 0) {
                    $this->addTranslatedMessage('Número de identificación ya se encuentra registrado', 'danger');
                } else {
                    $em->persist($transportista);
                    $em->flush();

                    $this->addTranslatedMessage('Registro actualizado');
                    return $this->redirectToRoute('transportista_index');
                }
            } else {
                $this->addTranslatedMessage('Documento del Transportista NO Válido ', 'danger');
                //return $this->redirectToRoute('registro_nuevo');
            }
        }

        return $this->render('Transportista/edit.html.twig',
            ['form' => $form->createView(), 'entity' => $transportista]);
    }

    /**
     * @Route("/edit/{slug}tra{id}", name="transportista_edit")
     *
     */
    public function editAction($id, EmFactory $emFactory, Request $request, $slug, TranslatorInterface $translator): \Symfony\Component\HttpFoundation\Response
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $transportista = $em->getRepository(Transportista::class)->findBySlug($slug, $id);
        if (!$transportista) {
            throw $this->createNotFoundException('Unable to find Transportista entity.');
        }


        $form = $this->createForm(TransportistaType::class, $transportista, [
            'action' => $this->generateUrl('transportista_edit', ['slug' => $slug, 'id' => $id])
        ]);
        $form->handleRequest($request);

        if ($form->isSubmitted()) {
            if ($request->request->has('Form-delete')) {
                $em->remove($transportista);
                $em->flush();
                $this->addTranslatedMessage('Registro Eliminado');
                return $this->redirectToRoute('transportista_index');
            }

            if ($form->isValid()) {
                $numDocumento = $transportista->getIdentification();

                $respuesta = null;

                //if ($transportista->getIdentification() != '9999999999999')
                //    $respuesta = $transportista->validarRuc($numDocumento, $transportista->getTipoDocumento());
                //else

                if ($respuesta === null) {
                    $transportista->setEmpresa($empresa);
                    $em->persist($transportista);
                    $em->flush();
                    $this->addTranslatedMessage('Registro Actualizado');

                    return $this->redirectToRoute('transportista_edit', ['slug' => $slug, 'id' => $id]);

                } else {
                    $this->addTranslatedMessage('Documento del Transportista NO Válido ', 'danger');
                    //return $this->redirectToRoute('registro_nuevo');
                }
            }
        }

        return $this->render('Transportista/edit.html.twig',
            ['form' => $form->createView(), 'entity' => $transportista]);
    }

    protected function addTranslatedMessage($message, $status = 'success')
    {
        $this->addFlash($status, $this->translator->trans($message, [], 'transportista'));

    }


    /**
     * @Route("/searchcliente", name="guia_transportista_search")
     */
    public function searchTransportistaAction(Request $request, EmFactory $emFactory): \Symfony\Component\HttpFoundation\Response
    {
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $entities = $em
            ->getRepository(Transportista::class)
            ->findIdentificacion($request->get('term'), $empresa);

        if (count($entities) > 0) {

            if (count($entities) > 1) {
                return new JsonResponse('ERROR, Se encontraron mas de un registro con el número de identificación ingresado',
                    \Symfony\Component\HttpFoundation\Response::HTTP_BAD_REQUEST);
            } else {
                return new JsonResponse(['message' => 'Transportista', 'item' => $entities[0]->jsonSerialize()], \Symfony\Component\HttpFoundation\Response::HTTP_OK);
            }

        } else {
            return new JsonResponse(
                'ERROR, Transportista no existe'
                , \Symfony\Component\HttpFoundation\Response::HTTP_BAD_REQUEST);
        }

    }

    /**
     * @Route("/rest/add", name="rest_transportista_add")
     */
    public function addTransportistaAction(Request $request, EmFactory $emFactory, ValidatorInterface $validator): \Symfony\Component\HttpFoundation\Response
    {
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $transportista = new Transportista();

        $form = $this->createForm(TransportistaType::class, $transportista, [
            'action' => $this->generateUrl('transportista_add'),
        ]);
        $form->handleRequest($request);

        if ($form->isSubmitted()) {
            $transportista->setEmpresa($empresa);
            $errors = $validator->validate($transportista);
            if (count($errors) > 0)
                return new JsonResponse(RestApiFunciones::getErrorValidacion($errors), \Symfony\Component\HttpFoundation\Response::HTTP_UNAUTHORIZED);
            else {
                $numDocumento = $transportista->getIdentification();
                //$respuesta = $transportista->validarRuc($numDocumento, $transportista->getTipoDocumento());
                $respuesta = null;
                if ($respuesta === null) {
                    $entities = $em->getRepository(Transportista::class)->findIdentificacion($numDocumento, $empresa);

                    if ((is_countable($entities) ? count($entities) : 0) > 0) {
                        return new JsonResponse('ERROR, YA SE ENCUENTRA REGISTRADO EL NUMERO DE DOCUMENTO: ' . $numDocumento,
                            \Symfony\Component\HttpFoundation\Response::HTTP_BAD_REQUEST);
                    }

                    try {
                        $em->persist($transportista);
                        $em->flush();

                        return new JsonResponse(['message' => 'Transportista', 'item' => $transportista->jsonSerialize()], \Symfony\Component\HttpFoundation\Response::HTTP_OK);
                    } catch (\Symfony\Component\Config\Definition\Exception\Exception $exception) {
                        return new JsonResponse('ERROR, ' . $exception->getMessage(),
                            \Symfony\Component\HttpFoundation\Response::HTTP_BAD_REQUEST);
                    }

                } else {
                    return new JsonResponse('ERROR, Documento del Transportista NO Válido ',
                        \Symfony\Component\HttpFoundation\Response::HTTP_BAD_REQUEST);
                }
            }
        }

        return new JsonResponse('ERROR, Desconocido consulte con el administrador ',
            \Symfony\Component\HttpFoundation\Response::HTTP_BAD_REQUEST);
    }

    /**
     * @Route("/search-reference", name="transportista_search_reference")
     */
    public function searchReferenceAction(EmFactory $emFactory, Request $request): \Symfony\Component\HttpFoundation\Response
    {
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');

        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $limit = 10;

        $aux = $request->request->all();

        $page = (int) $aux["page"];

        $start = $page > 1 ? ($page - 1) * $limit : 0;

        $condition = Funciones::cleanParametro($aux["query"]);

        $tipo = Funciones::cleanParametro($aux["tipo"]);

        //$entities = $em->getRepository('App\Entity\Model\Inventarios\Product')
        //    ->findxReference($request->get('term'), $empresa->getId());

        $repo = $em->getRepository(Transportista::class);

        $sql = "'p.id','p.reference','p.description'";
        $paginator = $repo->paginatedAjax($sql, $condition, $limit, $start, $empresa->getId());

        $total_data = is_countable($paginator) ? count($paginator) : 0;
        $pagination = [];
        foreach ($paginator as $post) {
            //$post = new Provider();
            $pagination[] = [
                'id' => $post->getIdentification(),
                'nombre' => $post->getName(),
                'email' => $post->getEmail(),
                'placa' => $post->getPlaca(),
                /*'address' => $post->getAddress()*/
            ];
        }

        if ($total_data > 0)
            $pagination_html = Funciones::getPaginationResults($total_data, $limit, $page, $condition, $tipo);
        else
            $pagination_html = '';

        $output = ['data' => $pagination, 'pagination' => $pagination_html, 'total_data' => $total_data];

        return new JsonResponse($output);

    }


}
